/*
 * Decompiled with CFR 0.152.
 */
package mil.army.usace.hec.data.timeseries.math;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import mil.army.usace.hec.data.timeseries.DataSetTimeSeriesAnalysisException;
import mil.army.usace.hec.data.timeseries.Quality;
import mil.army.usace.hec.data.timeseries.TimeSeries;
import mil.army.usace.hec.data.timeseries.TimeSeriesBuilder;
import mil.army.usace.hec.metadata.DataSetException;
import mil.army.usace.hec.metadata.UnitUtil;
import mil.army.usace.hec.metadata.Units;

public final class TimeSeriesAnalyzer {
    private static final Logger LOGGER = Logger.getLogger(TimeSeriesAnalyzer.class.getName());
    static final double DEFAULT_TOLERANCE = 1.0E-6;
    static final String THRESHOLD_PROPERTY = "hec.timeseries.analyze.time_window_threshold";
    static final int DEFAULT_THRESHOLD = Math.max(0, Integer.getInteger("hec.timeseries.analyze.time_window_threshold", 12));

    private TimeSeriesAnalyzer() {
        throw new AssertionError((Object)"Don't instantiate utility classes");
    }

    public static List<TimeWindow> getModifiedTimes(TimeSeries original, TimeSeries modified) throws DataSetTimeSeriesAnalysisException {
        return TimeSeriesAnalyzer.getModifiedTimes(original, modified, DEFAULT_THRESHOLD, 1.0E-6, original.getUnits());
    }

    public static List<TimeWindow> getModifiedTimes(TimeSeries original, TimeSeries modified, int timeWindowThreshold, double tolerance, Units toleranceUnits) throws DataSetTimeSeriesAnalysisException {
        Units realToleranceUnit = toleranceUnits;
        if (toleranceUnits == null) {
            realToleranceUnit = original.getUnits();
        }
        if (!TimeSeriesAnalyzer.canAnalyze(original, modified, realToleranceUnit)) {
            throw new DataSetTimeSeriesAnalysisException("Cannot analyze modified time window of:\n" + original.toString() + "\nWith:\n" + modified.toString() + "\nWith tolerance unit:\n" + toleranceUnits);
        }
        TimeSeries realOriginal = TimeSeriesAnalyzer.convertTsUnitsIfNecessary(original, realToleranceUnit);
        TimeSeries realModified = TimeSeriesAnalyzer.convertTsUnitsIfNecessary(modified, realToleranceUnit);
        return TimeSeriesAnalyzer.getModifiedTimesInternal(realOriginal, realModified, timeWindowThreshold, tolerance);
    }

    private static boolean canAnalyze(TimeSeries left, TimeSeries right, Units toleranceUnit) {
        boolean idEqual = left.getTimeSeriesIdentifier().equals(right.getTimeSeriesIdentifier());
        boolean unitsAreValid = TimeSeriesAnalyzer.canConvertToToleranceUnit(left, toleranceUnit) && TimeSeriesAnalyzer.canConvertToToleranceUnit(right, toleranceUnit);
        return idEqual && unitsAreValid;
    }

    private static boolean canConvertToToleranceUnit(TimeSeries ts, Units toleranceUnit) {
        boolean output = ts.getUnits().equals((Object)toleranceUnit);
        if (!output) {
            output = UnitUtil.canConvertBetweenUnits((Units)ts.getUnits(), (Units)toleranceUnit);
        }
        return output;
    }

    private static List<TimeWindow> getModifiedTimesInternal(TimeSeries original, TimeSeries modified, int timeWindowThreshold, double tolerance) {
        long[] originalTimes = original.getTimes();
        double[] originalValues = original.getValues();
        Quality originalQuality = original.getQuality().orElse(new Quality(originalTimes.length));
        Map<Long, TVQ> originalTimeValueQuality = TimeSeriesAnalyzer.buildTimeValueQualityMap(originalTimes, originalValues, originalQuality);
        long[] modifiedTimes = modified.getTimes();
        double[] modifiedValues = modified.getValues();
        Quality modifiedQuality = modified.getQuality().orElse(new Quality(modifiedTimes.length));
        Map<Long, TVQ> modifiedTimeValueQuality = TimeSeriesAnalyzer.buildTimeValueQualityMap(modifiedTimes, modifiedValues, modifiedQuality);
        return TimeSeriesAnalyzer.findModifiedTimeWindows(tolerance, timeWindowThreshold, originalTimeValueQuality, modifiedTimeValueQuality, originalTimes, modifiedTimes);
    }

    private static Map<Long, TVQ> buildTimeValueQualityMap(long[] times, double[] values, Quality quality) {
        return IntStream.range(0, times.length).mapToObj(i -> new TVQ(times[i], values[i], quality.getIntegerAt(i))).collect(Collectors.toMap(tvq -> tvq._time, Function.identity(), (left, right) -> left, TreeMap::new));
    }

    private static Set<Long> identifyAllModifiedTimes(long[] originalTimes, long[] modifiedTimes) {
        TreeSet<Long> allTimes = new TreeSet<Long>();
        int originalLength = originalTimes.length;
        if (originalLength > 0) {
            long start = modifiedTimes[0];
            long end = modifiedTimes[modifiedTimes.length - 1];
            int startIndex = Math.abs(Arrays.binarySearch(originalTimes, start));
            if (startIndex < originalLength) {
                long time;
                for (int i = startIndex; i < originalLength && (time = originalTimes[i]) <= end; ++i) {
                    allTimes.add(time);
                }
            }
        }
        for (long modifiedTime : modifiedTimes) {
            allTimes.add(modifiedTime);
        }
        return allTimes;
    }

    private static List<TimeWindow> findModifiedTimeWindows(double tolerance, int timeWindowThreshold, Map<Long, TVQ> originalTimeValueQuality, Map<Long, TVQ> modifiedTimeValueQuality, long[] originalTimes, long[] modifiedTimes) {
        ArrayList<TimeWindow> output = new ArrayList<TimeWindow>();
        Set<Long> allTimes = TimeSeriesAnalyzer.identifyAllModifiedTimes(originalTimes, modifiedTimes);
        TimeWindow currentTimeWindow = null;
        int unmodifiedValueCount = 0;
        for (long time : allTimes) {
            if (TimeSeriesAnalyzer.isModified(time, originalTimeValueQuality, modifiedTimeValueQuality, tolerance)) {
                if (currentTimeWindow == null) {
                    currentTimeWindow = new TimeWindow();
                    output.add(currentTimeWindow);
                }
                unmodifiedValueCount = 0;
                currentTimeWindow.add(Instant.ofEpochMilli(time));
                continue;
            }
            if (unmodifiedValueCount >= timeWindowThreshold) {
                currentTimeWindow = null;
                unmodifiedValueCount = 0;
                continue;
            }
            ++unmodifiedValueCount;
        }
        return output;
    }

    private static boolean isModified(long time, Map<Long, TVQ> originalTimeValueQuality, Map<Long, TVQ> modifiedTimeValueQuality, double tolerance) {
        TVQ modifiedTvq;
        boolean output = true;
        TVQ originalTvq = originalTimeValueQuality.get(time);
        if (originalTvq != null && (modifiedTvq = modifiedTimeValueQuality.get(time)) != null) {
            double originalValue = originalTvq._value;
            double modifiedValue = modifiedTvq._value;
            int originalQualityVal = originalTvq._quality;
            int modifiedQualityVal = modifiedTvq._quality;
            output = originalQualityVal != modifiedQualityVal || Math.abs(originalValue - modifiedValue) > tolerance;
        }
        return output;
    }

    private static TimeSeries convertTsUnitsIfNecessary(TimeSeries original, Units toleranceUnit) throws DataSetTimeSeriesAnalysisException {
        TimeSeries realOriginal = original;
        try {
            if (original.needToChangeUnits(toleranceUnit)) {
                realOriginal = TimeSeriesBuilder.builder(original).build();
                realOriginal.changeUnits(toleranceUnit);
            }
        }
        catch (DataSetException ex) {
            throw new DataSetTimeSeriesAnalysisException("Unable to convert time series from " + realOriginal.getUnits().toString() + " units to " + toleranceUnit.toString(), ex);
        }
        return realOriginal;
    }

    static {
        LOGGER.config(() -> "Using " + DEFAULT_THRESHOLD + " for default threshold.\nSystem property hec.timeseries.analyze.time_window_threshold = " + System.getProperty(THRESHOLD_PROPERTY));
    }

    public static final class TimeWindow {
        private final NavigableSet<Instant> _discreteDates = new TreeSet<Instant>();
        private Instant _firstDate;
        private Instant _lastDate;

        private TimeWindow() {
        }

        private void add(Instant instant) {
            this._discreteDates.add(instant);
            if (this._firstDate == null || this._lastDate == null) {
                this._firstDate = instant;
                this._lastDate = instant;
            } else if (instant.isBefore(this._firstDate)) {
                this._firstDate = instant;
            } else if (instant.isAfter(this._lastDate)) {
                this._lastDate = instant;
            }
        }

        NavigableSet<Instant> getDiscreteDates() {
            return new TreeSet<Instant>((SortedSet<Instant>)this._discreteDates);
        }

        public Instant getFirstDate() {
            return this._firstDate;
        }

        public Instant getLastDate() {
            return this._lastDate;
        }
    }

    private static class TVQ {
        private final long _time;
        private final double _value;
        private final int _quality;

        public TVQ(long time, double value, int quality) {
            this._quality = quality;
            this._time = time;
            this._value = value;
        }
    }
}

