/*
 * Decompiled with CFR 0.152.
 */
package rma.lang;

import java.util.Arrays;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import rma.lang.RmaDerivFunction;
import rma.lang.RmaSVFunction;
import rma.util.RMAConst;

public class RmaMath {
    private static final double rk2PGROW = -0.333333333;
    private static final double rk2PSHRINK = -0.5;
    private static final double rk2FCOR = 0.333333333;
    private static final double rk2SAFETY = 0.9;
    private static final double rk2ERRCON = 0.011;
    private static final Logger LOGGER = Logger.getLogger(RmaMath.class.getName());
    public static int gdISET = 0;
    public static double gdGSET = 0.0;

    public static double gasdev() {
        if (gdISET == 0) {
            double v1 = 0.0;
            double v2 = 0.0;
            double r = 10.0;
            for (int icnt = 0; (r > 1.0 || r == 0.0) && icnt < 10000; ++icnt) {
                v1 = 2.0 * Math.random() - 1.0;
                v2 = 2.0 * Math.random() - 1.0;
                r = v1 * v1 + v2 * v2;
            }
            double fac = Math.sqrt(-2.0 * Math.log(r) / r);
            gdGSET = v1 * fac;
            gdISET = 1;
            return v2 * fac;
        }
        gdISET = 0;
        return gdGSET;
    }

    public static int odeint2(double[] ystart, double x1, double x2, double eps, double h1, double hmin, int[] nok, int[] nbad, RmaDerivFunction derivfunc, int maxsteps, double[] xp, double[][] yp, double dxsav, int[] kount) {
        int i;
        if (ystart == null || nok == null || nbad == null || derivfunc == null) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "odeint2: error - invalid input array");
            }
            return -3;
        }
        int nvar = ystart.length;
        double[] x = new double[1];
        double[] hnext = new double[1];
        double[] hdid = new double[1];
        double[] yscl = new double[nvar];
        double[] y = new double[nvar];
        double[] dydx = new double[nvar];
        double[] ysav = new double[nvar];
        double[] dysav = new double[nvar];
        double[] ytmp = new double[nvar];
        double[] dytmp = new double[nvar];
        int rtn = 0;
        boolean[] clipAtMin = new boolean[nvar];
        int kmax = 0;
        if (xp != null) {
            kmax = xp.length;
            kount[0] = 0;
        }
        x[0] = x1;
        double xsav = x[0];
        double h = x2 > x1 ? Math.abs(h1) : -Math.abs(h1);
        nbad[0] = 0;
        nok[0] = 0;
        for (i = 0; i < nvar; ++i) {
            y[i] = ystart[i];
        }
        if (kmax > 0) {
            xsav = x[0] - dxsav * 2.0;
        }
        for (int nstp = 0; nstp < maxsteps; ++nstp) {
            derivfunc.evaluate(x[0], y, dydx, h * 0.5, y);
            for (i = 0; i < nvar; ++i) {
                yscl[i] = Math.abs(y[i]) + Math.abs(dydx[i] * h) + Double.MIN_VALUE;
            }
            if (kmax > 0 && Math.abs(x[0] - xsav) > Math.abs(dxsav) && kount[0] < kmax) {
                xp[kount[0]] = x[0];
                for (i = 0; i < nvar; ++i) {
                    yp[i][kount[0]] = y[i];
                }
                xsav = x[0];
                kount[0] = kount[0] + 1;
            }
            if ((x[0] + h - x2) * (x[0] + h - x1) > 0.0) {
                h = x2 - x[0];
            }
            RmaMath.rk2qcStep(y, dydx, x, h, eps, yscl, hdid, hnext, derivfunc, maxsteps - nstp, ysav, dysav, ytmp, dytmp);
            if (hdid[0] == h) {
                nok[0] = nok[0] + 1;
            } else {
                nbad[0] = nbad[0] + 1;
            }
            if ((x[0] - x2) * (x2 - x1) >= 0.0) {
                for (i = 0; i < nvar; ++i) {
                    ystart[i] = y[i];
                }
                if (kmax != 0) {
                    xp[kount[0]] = x[0];
                    for (i = 0; i < nvar; ++i) {
                        yp[i][kount[0]] = y[i];
                    }
                    kount[0] = kount[0] + 1;
                }
                return rtn;
            }
            if (Math.abs(hnext[0]) <= hmin) {
                hnext[0] = hmin;
                rtn = -1;
            }
            h = hnext[0];
        }
        return -2;
    }

    public static void rk2qcStep(double[] y, double[] dydx, double[] x, double htry, double eps, double[] yscl, double[] hdid, double[] hnext, RmaDerivFunction derivfunc, int maxintv, double[] ysav, double[] dysav, double[] ytmp, double[] dytmp) {
        int i;
        int imax = y.length;
        double xsav = x[0];
        for (i = 0; i < imax; ++i) {
            ysav[i] = y[i];
            dysav[i] = dydx[i];
        }
        double h = htry;
        int icnt = 0;
        while (icnt++ < maxintv) {
            double hh = 0.5 * h;
            RmaMath.rungeKutta2(ysav, dysav, xsav, hh, derivfunc, ytmp, dytmp);
            x[0] = xsav + hh;
            derivfunc.evaluate(x[0], ytmp, dydx, hh, ytmp);
            boolean slopeOK = true;
            for (i = 0; i < imax; ++i) {
                if (!(dysav[i] > dytmp[i] && dydx[i] > dytmp[i]) && (!(dysav[i] < dytmp[i]) || !(dydx[i] < dytmp[i]))) continue;
                slopeOK = false;
            }
            RmaMath.rungeKutta2(ytmp, dydx, x[0], hh, derivfunc, y, dytmp);
            x[0] = xsav + h;
            if (x[0] == xsav) {
                return;
            }
            RmaMath.rungeKutta2(ysav, dysav, xsav, h, derivfunc, ytmp, dytmp);
            double errmax = 0.0;
            for (i = 0; i < imax; ++i) {
                ytmp[i] = y[i] - ytmp[i];
                double err = Math.abs(ytmp[i] / yscl[i]);
                if (!(errmax < err)) continue;
                errmax = err;
            }
            if ((errmax /= eps) <= 1.0 && slopeOK) {
                hdid[0] = h;
                hnext[0] = errmax > 0.011 ? 0.9 * h * Math.exp(-0.333333333 * Math.log(errmax)) : 4.0 * h;
                break;
            }
            if (!slopeOK) {
                h *= 0.5;
                continue;
            }
            double hfac = 0.9 * Math.exp(-0.5 * Math.log(errmax));
            if (hfac < 0.2) {
                hfac = 0.2;
            }
            h *= hfac;
        }
        for (i = 0; i < imax; ++i) {
            int n = i;
            y[n] = y[n] + ytmp[i] * 0.333333333;
        }
    }

    public static void rungeKutta2(double[] y0, double[] dydx, double x, double h, RmaDerivFunction derivfunc, double[] y1, double[] dydxm) {
        int i;
        int imax = y0.length;
        for (i = 0; i < imax; ++i) {
            y1[i] = y0[i] + dydx[i] * h * 0.5;
        }
        double xm = x + h * 0.5;
        derivfunc.evaluate(xm, y1, dydxm, h, y0);
        for (i = 0; i < imax; ++i) {
            y1[i] = y0[i] + dydxm[i] * h;
        }
    }

    public static double fpRoot(RmaSVFunction func, double x0, double x1, int maxiter, double tol) {
        double x = Double.NEGATIVE_INFINITY;
        double fx = Double.NEGATIVE_INFINITY;
        double fx0 = func.evaluate(x0);
        double fx1 = func.evaluate(x1);
        if (Math.abs(fx0) < tol) {
            return x0;
        }
        if (Math.abs(fx1) < tol) {
            return x1;
        }
        if (fx1 * fx0 >= 0.0) {
            return Double.NEGATIVE_INFINITY;
        }
        if (fx0 > fx1) {
            x = x0;
            x0 = x1;
            x1 = x;
            fx = fx0;
            fx0 = fx1;
            fx1 = fx;
        }
        for (int i = 0; i < maxiter; ++i) {
            x = x0 - fx0 * ((x1 - x0) / (fx1 - fx0));
            if (x <= x0 || x >= x1) {
                x = (x0 + x1) * 0.5;
            }
            if (Math.abs(fx = func.evaluate(x)) < tol) {
                if (LOGGER.isLoggable(Level.INFO)) {
                    LOGGER.log(Level.INFO, "fpRoot: x=" + x + " fx=" + fx + " i=" + i);
                }
                return x;
            }
            if (fx > fx1) {
                fx0 = fx1;
                x0 = x1;
                fx1 = fx;
                x1 = x;
                continue;
            }
            if (fx < fx0) {
                fx1 = fx0;
                x1 = x0;
                fx0 = fx;
                x0 = x;
                continue;
            }
            if (fx > 0.0) {
                fx1 = fx;
                x1 = x;
                continue;
            }
            fx0 = fx;
            x0 = x;
        }
        return RmaMath.bisectRoot(func, x0, x1, maxiter, tol);
    }

    public static double bisectRoot(RmaSVFunction func, double x0, double x1, int maxiter, double tol) {
        int i;
        double x = Double.NEGATIVE_INFINITY;
        double fx = Double.NEGATIVE_INFINITY;
        double fx0 = func.evaluate(x0);
        double fx1 = func.evaluate(x1);
        if (Math.abs(fx0) < tol) {
            return x0;
        }
        if (Math.abs(fx1) < tol) {
            return x1;
        }
        if (fx1 * fx0 >= 0.0) {
            return Double.NEGATIVE_INFINITY;
        }
        if (fx0 > fx1) {
            x = x0;
            x0 = x1;
            x1 = x;
            fx = fx0;
            fx0 = fx1;
            fx1 = fx;
        }
        for (i = 0; i < maxiter; ++i) {
            x = (x0 + x1) * 0.5;
            fx = func.evaluate(x);
            if (Math.abs(fx) < tol) {
                return x;
            }
            if (fx > fx1) {
                fx1 = fx;
                x1 = x;
                continue;
            }
            if (fx < fx0) {
                fx0 = fx;
                x0 = x;
                continue;
            }
            if (fx > 0.0) {
                fx1 = fx;
                x1 = x;
                continue;
            }
            fx0 = fx;
            x0 = x;
        }
        func.printError("bisectRoot: x=" + x + " fx=" + fx + " i=" + i + " NC");
        return x;
    }

    public static double BrentsMethodSolve(RmaSVFunction function, double lowerLimit, double upperLimit, int maxiter, double errorTol) {
        double a = lowerLimit;
        double b = upperLimit;
        double c2 = 0.0;
        double d = Double.MAX_VALUE;
        double fa = function.evaluate(a);
        double fb = function.evaluate(b);
        double fc = 0.0;
        double s = 0.0;
        double fs = 0.0;
        if (fa * fb >= 0.0) {
            if (fa < fb) {
                return a;
            }
            return b;
        }
        if (Math.abs(fa) < Math.abs(fb)) {
            double tmp = a;
            a = b;
            b = tmp;
            tmp = fa;
            fa = fb;
            fb = tmp;
        }
        c2 = a;
        fc = fa;
        boolean mflag = true;
        int i = 0;
        while (fb != 0.0 && Math.abs(a - b) > errorTol) {
            double tmp2;
            s = fa != fc && fb != fc ? a * fb * fc / (fa - fb) / (fa - fc) + b * fa * fc / (fb - fa) / (fb - fc) + c2 * fa * fb / (fc - fa) / (fc - fb) : b - fb * (b - a) / (fb - fa);
            if ((!(s > (tmp2 = (3.0 * a + b) / 4.0)) || !(s < b)) && (!(s < tmp2) || !(s > b)) || mflag && Math.abs(s - b) >= Math.abs(b - c2) / 2.0 || !mflag && Math.abs(s - b) >= Math.abs(c2 - d) / 2.0) {
                s = (a + b) / 2.0;
                mflag = true;
            } else if (mflag && Math.abs(b - c2) < errorTol || !mflag && Math.abs(c2 - d) < errorTol) {
                s = (a + b) / 2.0;
                mflag = true;
            } else {
                mflag = false;
            }
            fs = function.evaluate(s);
            d = c2;
            c2 = b;
            fc = fb;
            if (fa * fs < 0.0) {
                b = s;
                fb = fs;
            } else {
                a = s;
                fa = fs;
            }
            if (Math.abs(fa) < Math.abs(fb)) {
                double tmp = a;
                a = b;
                b = tmp;
                tmp = fa;
                fa = fb;
                fb = tmp;
            }
            if (++i <= maxiter) continue;
            function.printError("BrentsMethodSolve: exceeded max iterations, Error is " + fb);
            break;
        }
        return b;
    }

    public static double balancedDistribution(double target, double[] dist, double[] weight, double[] min, double[] max, double tolfac, boolean[] active, boolean[] minlimited, boolean[] maxlimited) {
        int i;
        if (dist == null || dist.length < 1) {
            return Double.NEGATIVE_INFINITY;
        }
        int numval = dist.length;
        if (weight == null || weight.length < numval) {
            weight = new double[numval];
            Arrays.fill(weight, 1.0);
        }
        if (min == null || min.length < numval) {
            min = new double[numval];
            Arrays.fill(min, Double.MIN_VALUE);
        }
        if (max == null || max.length < numval) {
            max = new double[numval];
            Arrays.fill(max, Double.MAX_VALUE);
        }
        if (active == null || active.length < numval) {
            active = new boolean[numval];
            Arrays.fill(active, true);
        }
        if (!RMAConst.isValidValue(tolfac)) {
            tolfac = 0.0;
        }
        if (minlimited == null) {
            minlimited = new boolean[numval];
        }
        if (maxlimited == null) {
            maxlimited = new boolean[numval];
        }
        double sumwgt = 0.0;
        double summin = 0.0;
        double summax = 0.0;
        double sumdist = 0.0;
        int numactive = 0;
        double tol = tolfac * target;
        for (i = 0; i < numval; ++i) {
            if (!active[i]) continue;
            sumwgt += weight[i];
            summin += min[i];
            summax += max[i];
            ++numactive;
        }
        if (sumwgt - tol <= 0.0) {
            return Double.NEGATIVE_INFINITY;
        }
        if (summin + tol >= target) {
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                dist[i] = min[i];
                minlimited[i] = true;
                maxlimited[i] = dist[i] + tol > max[i];
            }
            sumdist = summin;
        } else if (summax - tol <= target) {
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                dist[i] = max[i];
                maxlimited[i] = true;
                minlimited[i] = dist[i] - tol < min[i];
            }
            sumdist = summax;
        } else {
            Arrays.fill(maxlimited, false);
            double remaining = target;
            int maxcnt = 0;
            while (maxcnt++ < numval) {
                int numActive = 0;
                double remainingwgt = 0.0;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || maxlimited[i]) continue;
                    minlimited[i] = false;
                    remainingwgt += weight[i];
                    ++numActive;
                }
                if (numActive == 0) break;
                boolean done = false;
                double tmpRemaining = remaining;
                double tmpRemainWgt = remainingwgt;
                int icnt = 0;
                while (icnt++ < numval) {
                    double curtarget = tmpRemaining;
                    double cursumwgt = tmpRemainWgt;
                    sumdist = target - tmpRemaining;
                    for (i = 0; i < numval; ++i) {
                        if (!active[i] || maxlimited[i] || minlimited[i]) continue;
                        double val = curtarget * weight[i] / cursumwgt;
                        if (val - tol <= min[i]) {
                            dist[i] = min[i];
                            tmpRemaining -= dist[i];
                            tmpRemainWgt -= weight[i];
                            minlimited[i] = true;
                        } else {
                            dist[i] = val;
                        }
                        sumdist += dist[i];
                    }
                    if (!(Math.abs(sumdist - target) <= tol) && !(remainingwgt <= 0.0)) continue;
                }
                int ilargest = -1;
                double violation = 0.0;
                done = true;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || maxlimited[i] || !(dist[i] - max[i] > violation)) continue;
                    violation = dist[i] - max[i];
                    ilargest = i;
                }
                if (ilargest >= 0) {
                    dist[ilargest] = max[ilargest];
                    maxlimited[ilargest] = true;
                    remaining -= dist[ilargest];
                    remainingwgt -= weight[ilargest];
                    done = false;
                }
                if (!done && !(remainingwgt <= 0.0)) continue;
                break;
            }
            sumdist = 0.0;
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                sumdist += dist[i];
            }
        }
        return sumdist;
    }

    public static double balancedDistWithLoss(double target, double[] dist, double[] weight, double[] min, double[] max, double tolfac, boolean[] active, boolean[] minlimited, boolean[] maxlimited, double[] lossfac) {
        int i;
        if (dist == null || dist.length < 1) {
            return Double.NEGATIVE_INFINITY;
        }
        int numval = dist.length;
        if (weight == null || weight.length < numval) {
            weight = new double[numval];
            Arrays.fill(weight, 1.0);
        }
        if (min == null || min.length < numval) {
            min = new double[numval];
            Arrays.fill(min, Double.MIN_VALUE);
        }
        if (max == null || max.length < numval) {
            max = new double[numval];
            Arrays.fill(max, Double.MAX_VALUE);
        }
        if (lossfac == null || lossfac.length < numval) {
            lossfac = new double[numval];
            Arrays.fill(lossfac, 1.0);
        }
        if (active == null || active.length < numval) {
            active = new boolean[numval];
            Arrays.fill(active, true);
        }
        if (!RMAConst.isValidValue(tolfac)) {
            tolfac = 0.0;
        }
        if (minlimited == null) {
            minlimited = new boolean[numval];
        }
        if (maxlimited == null) {
            maxlimited = new boolean[numval];
        }
        double sumwgt = 0.0;
        double summin = 0.0;
        double summax = 0.0;
        double sumdist = 0.0;
        int numactive = 0;
        double tol = tolfac * target;
        for (i = 0; i < numval; ++i) {
            if (!active[i]) continue;
            sumwgt += weight[i] * lossfac[i];
            summin += min[i] * lossfac[i];
            summax += max[i] * lossfac[i];
            ++numactive;
        }
        if (sumwgt - tol <= 0.0) {
            return Double.NEGATIVE_INFINITY;
        }
        if (summin + tol >= target) {
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                dist[i] = min[i];
                minlimited[i] = true;
                maxlimited[i] = dist[i] + tol > max[i];
            }
            sumdist = summin;
        } else if (summax - tol <= target) {
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                dist[i] = max[i];
                maxlimited[i] = true;
                minlimited[i] = dist[i] - tol < min[i];
            }
            sumdist = summax;
        } else {
            double val;
            Arrays.fill(minlimited, false);
            Arrays.fill(maxlimited, false);
            boolean done = false;
            double remaining = target;
            double remainingwgt = sumwgt;
            int icnt = 0;
            while (icnt++ < numval) {
                double curtarget = remaining;
                double cursumwgt = remainingwgt;
                sumdist = target - remaining;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || minlimited[i]) continue;
                    val = curtarget * weight[i] / cursumwgt;
                    if (val - tol < min[i]) {
                        dist[i] = min[i];
                        remaining -= dist[i] * lossfac[i];
                        remainingwgt -= weight[i] * lossfac[i];
                        minlimited[i] = true;
                    } else {
                        dist[i] = val;
                    }
                    sumdist += dist[i] * lossfac[i];
                }
                if (!(Math.abs(sumdist - target) <= tol) && !(remainingwgt <= 0.0)) continue;
            }
            done = false;
            icnt = 0;
            remaining = target;
            remainingwgt = sumwgt;
            while (icnt++ < numval) {
                done = true;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || maxlimited[i] || !(dist[i] + tol > max[i])) continue;
                    dist[i] = max[i];
                    maxlimited[i] = true;
                    remaining -= dist[i] * lossfac[i];
                    remainingwgt -= weight[i] * lossfac[i];
                    done = false;
                }
                if (done || remainingwgt <= 0.0) break;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || maxlimited[i]) continue;
                    val = remaining * weight[i] / remainingwgt;
                    dist[i] = val - tol < min[i] ? min[i] : val;
                }
            }
            sumdist = 0.0;
            Arrays.fill(minlimited, false);
            for (i = 0; i < numval; ++i) {
                if (!active[i]) continue;
                sumdist += dist[i] * lossfac[i];
                if (!(dist[i] - tol <= min[i])) continue;
                minlimited[i] = true;
            }
        }
        return sumdist;
    }

    public static double redistributeWithConstraint(double total, double[] origDist, double[] dist, double[] min, double[] max, double tolfac, boolean[] active) {
        double abvmax;
        int i;
        if (origDist == null || origDist.length < 1 || dist == null || dist.length < 1) {
            return Double.NEGATIVE_INFINITY;
        }
        int numval = origDist.length;
        if (min == null || min.length < numval) {
            min = new double[numval];
            Arrays.fill(min, Double.MIN_VALUE);
        }
        if (max == null || max.length < numval) {
            max = new double[numval];
            Arrays.fill(max, Double.MAX_VALUE);
        }
        if (active == null || active.length < numval) {
            active = new boolean[numval];
            Arrays.fill(active, true);
        }
        if (!RMAConst.isValidValue(tolfac)) {
            tolfac = 0.0;
        }
        double sumdist = 0.0;
        int numactive = 0;
        for (i = 0; i < numval; ++i) {
            dist[i] = origDist[i];
            if (!active[i]) continue;
            ++numactive;
            if (!(origDist[i] > 0.0)) continue;
            sumdist += origDist[i];
        }
        if (numactive == 0) {
            return Double.NEGATIVE_INFINITY;
        }
        for (i = 0; i < numval; ++i) {
            int j;
            if (!active[i] || !(dist[i] < 0.0)) continue;
            double redistVal = -dist[i];
            double sum = 0.0;
            for (j = 0; j < numval; ++j) {
                if (!active[j] || i == j || !(dist[j] > max[j])) continue;
                sum += dist[j] - max[j];
            }
            if (sum >= redistVal) {
                for (j = 0; j < numval; ++j) {
                    if (!active[j] || i == j || !(dist[j] > max[j])) continue;
                    int n = j;
                    dist[n] = dist[n] - redistVal * (dist[j] - max[j]) / sum;
                }
                redistVal = 0.0;
                dist[i] = 0.0;
            } else if (sum > 0.0) {
                for (j = 0; j < numval; ++j) {
                    if (!active[j] || i == j || !(dist[j] > max[j])) continue;
                    dist[j] = max[j];
                }
                redistVal -= sum;
                int n = i;
                dist[n] = dist[n] + sum;
            }
            if (redistVal <= 0.0) continue;
            sum = 0.0;
            for (j = 0; j < numval; ++j) {
                if (!active[j] || i == j || !(dist[j] > min[j])) continue;
                sum += dist[j] - min[j];
            }
            if (sum > redistVal) {
                for (j = 0; j < numval; ++j) {
                    if (!active[j] || i == j || !(dist[j] > min[j])) continue;
                    dist[j] = redistVal * (dist[j] - min[j]) / sum + dist[j];
                }
                redistVal = 0.0;
                dist[i] = 0.0;
            } else if (sum > 0.0) {
                for (j = 0; j < numval; ++j) {
                    if (!active[j] || i == j || !(dist[j] > min[j])) continue;
                    dist[j] = min[j];
                }
                redistVal -= sum;
                int n = i;
                dist[n] = dist[n] + sum;
            }
            if (redistVal <= 0.0) continue;
            sum = 0.0;
            for (j = 0; j < numval; ++j) {
                if (!active[j] || i == j || dist[j] <= 0.0 || !(dist[j] > 0.0)) continue;
                sum += dist[j];
            }
            if (sum > redistVal) {
                for (j = 0; j < numval; ++j) {
                    if (!active[j] || i == j || !(dist[j] > 0.0)) continue;
                    dist[j] = redistVal * dist[j] / sum;
                }
                redistVal = 0.0;
                dist[i] = 0.0;
                continue;
            }
            if (!(sum > 0.0)) continue;
            for (j = 0; j < numval; ++j) {
                if (!active[j] || i == j || !(dist[j] > 0.0)) continue;
                dist[j] = 0.0;
            }
            redistVal -= sum;
            int n = i;
            dist[n] = dist[n] + sum;
        }
        sumdist = 0.0;
        for (i = 0; i < numval; ++i) {
            if (!active[i]) continue;
            sumdist += dist[i];
        }
        if (sumdist <= 0.0) {
            return Double.NEGATIVE_INFINITY;
        }
        for (i = 0; i < numval; ++i) {
            if (!active[i]) continue;
            int n = i;
            dist[n] = dist[n] * (total / sumdist);
        }
        double blwmin = 0.0;
        for (i = 0; i < numval; ++i) {
            if (!active[i] || !(dist[i] < min[i])) continue;
            blwmin += min[i] - dist[i];
        }
        if (blwmin > 0.0) {
            abvmax = 0.0;
            for (i = 0; i < numval; ++i) {
                if (!active[i] || !(dist[i] > max[i])) continue;
                abvmax += max[i] - dist[i];
            }
            if (abvmax > blwmin) {
                for (i = 0; i < numval; ++i) {
                    if (!active[i]) continue;
                    if (dist[i] < min[i]) {
                        dist[i] = min[i];
                        continue;
                    }
                    if (!(dist[i] > max[i])) continue;
                    dist[i] = dist[i] - (max[i] - dist[i]) * blwmin / abvmax;
                }
                blwmin = 0.0;
            } else if (abvmax > 0.0) {
                for (i = 0; i < numval; ++i) {
                    if (!active[i]) continue;
                    if (dist[i] > max[i]) {
                        dist[i] = max[i];
                        continue;
                    }
                    if (!(dist[i] < min[i])) continue;
                    dist[i] = dist[i] + (min[i] - dist[i]) * abvmax / blwmin;
                }
                blwmin -= abvmax;
            }
            if (blwmin > 0.0) {
                double abvmin = 0.0;
                for (i = 0; i < numval; ++i) {
                    if (!active[i] || !(dist[i] > min[i])) continue;
                    abvmin += dist[i] - min[i];
                }
                if (abvmin > blwmin) {
                    for (i = 0; i < numval; ++i) {
                        if (!active[i]) continue;
                        if (dist[i] < min[i]) {
                            dist[i] = min[i];
                            continue;
                        }
                        if (!(dist[i] > min[i])) continue;
                        dist[i] = dist[i] - (dist[i] - min[i]) * blwmin / abvmin;
                    }
                    blwmin = 0.0;
                } else if (abvmin > 0.0) {
                    for (i = 0; i < numval; ++i) {
                        if (!active[i]) continue;
                        if (dist[i] > min[i]) {
                            dist[i] = min[i];
                            continue;
                        }
                        if (!(dist[i] < min[i])) continue;
                        dist[i] = dist[i] + (min[i] - dist[i]) * abvmin / blwmin;
                    }
                    blwmin -= abvmin;
                }
            }
        }
        abvmax = 0.0;
        for (i = 0; i < numval; ++i) {
            if (!active[i] || !(dist[i] > max[i])) continue;
            abvmax += dist[i] - max[i];
        }
        if (abvmax > 0.0) {
            double blwmax = 0.0;
            for (i = 0; i < numval; ++i) {
                if (!active[i] || !(dist[i] < max[i])) continue;
                blwmax += max[i] - dist[i];
            }
            if (blwmax > abvmax) {
                for (i = 0; i < numval; ++i) {
                    if (!active[i]) continue;
                    if (dist[i] > max[i]) {
                        dist[i] = max[i];
                        continue;
                    }
                    if (!(dist[i] < max[i])) continue;
                    dist[i] = dist[i] + (max[i] - dist[i]) * abvmax / blwmax;
                }
                abvmax = 0.0;
            } else if (blwmax > 0.0) {
                for (i = 0; i < numval; ++i) {
                    if (!active[i]) continue;
                    if (dist[i] < max[i]) {
                        dist[i] = max[i];
                        continue;
                    }
                    if (!(dist[i] > max[i])) continue;
                    dist[i] = dist[i] - (dist[i] - max[i]) * blwmax / abvmax;
                }
                abvmax -= blwmax;
            }
        }
        double cksum = 0.0;
        for (i = 0; i < numval; ++i) {
            if (!active[i]) continue;
            cksum += dist[i];
        }
        return cksum;
    }

    public static double[] normalize(double[] origValues, double sumValue) {
        if (origValues == null) {
            return null;
        }
        double[] normalValues = new double[origValues.length];
        double sum = 0.0;
        for (int i = 0; i < origValues.length; ++i) {
            sum += origValues[i];
        }
        double normalMultiplier = sumValue / sum;
        for (int i = 0; i < normalValues.length; ++i) {
            normalValues[i] = normalMultiplier * origValues[i];
        }
        return normalValues;
    }

    public static double linearInterpolate(double x1, double y1, double x2, double y2, double x) {
        double y = (y2 - y1) / (x2 - x1) * x + (y1 - (y2 - y1) / (x2 - x1) * x1);
        return y;
    }

    public static int mod(int i1, int i2) {
        double value = (double)i1 / (double)i2;
        int quotient = (int)Math.floor(value);
        return i1 - i2 * quotient;
    }

    public static double[] bestFitLinear(double[] x, double[] y, int ncoord, int istart) {
        if (x == null || y == null) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: Input parameter is null");
            }
            return null;
        }
        if (ncoord < 2) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: must at least 2 coordinates.");
            }
            return null;
        }
        if (x.length < ncoord || y.length < ncoord) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: Input must have same number of coordinates");
            }
            return null;
        }
        double[] result = new double[2];
        double ATA11 = 0.0;
        double ATA12 = 0.0;
        double ATA21 = 0.0;
        double ATA22 = 0.0;
        double ATc1 = 0.0;
        double ATc2 = 0.0;
        for (int i = istart; i < istart + ncoord; ++i) {
            ATA11 += x[i] * x[i];
            ATA12 += x[i];
            ATA22 += 1.0;
            ATc1 += x[i] * y[i];
            ATc2 += y[i];
        }
        ATA21 = ATA12;
        if (ATA12 == 0.0 || 1.0 - ATA22 * ATA11 / (ATA21 * ATA12) == 0.0) {
            result[0] = Double.NEGATIVE_INFINITY;
            result[1] = ATA12 / ATA22;
        } else {
            result[0] = (ATc2 - ATA22 * ATc1 / ATA12) / ATA21 / (1.0 - ATA22 * ATA11 / (ATA21 * ATA12));
            result[1] = (ATc1 - ATA11 * result[0]) / ATA12;
        }
        return result;
    }

    public static double[] bestFitLinear(int[] x, int[] y, int ncoord, int istart) {
        if (x == null || y == null) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: Input parameter is null");
            }
            return null;
        }
        if (ncoord < 2) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: must at least 2 coordinates.");
            }
            return null;
        }
        if (x.length < ncoord || y.length < ncoord) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - bestFitLinear: Input must have same number of coordinates");
            }
            return null;
        }
        double[] result = new double[2];
        double ATA11 = 0.0;
        double ATA12 = 0.0;
        double ATA21 = 0.0;
        double ATA22 = 0.0;
        double ATc1 = 0.0;
        double ATc2 = 0.0;
        for (int i = istart; i < istart + ncoord; ++i) {
            ATA11 += (double)(x[i] * x[i]);
            ATA12 += (double)x[i];
            ATA22 += 1.0;
            ATc1 += (double)(x[i] * y[i]);
            ATc2 += (double)y[i];
        }
        ATA21 = ATA12;
        if (ATA12 == 0.0 || 1.0 - ATA22 * ATA11 / (ATA21 * ATA12) == 0.0) {
            result[0] = Double.NEGATIVE_INFINITY;
            result[1] = ATA12 / ATA22;
        } else {
            result[0] = (ATc2 - ATA22 * ATc1 / ATA12) / ATA21 / (1.0 - ATA22 * ATA11 / (ATA21 * ATA12));
            result[1] = (ATc1 - ATA11 * result[0]) / ATA12;
        }
        return result;
    }

    public static double[] constructAffineTransform(double[] x, double[] y, double[] xp, double[] yp) {
        int ii;
        if (x == null || y == null || xp == null || yp == null) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - constructAffineTransform: Input parameter is null");
            }
            return null;
        }
        if (x.length < 3) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - constructAffineTransform: must at least 3 coordinates.");
            }
            return null;
        }
        if (x.length > y.length || x.length > xp.length || x.length > yp.length) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Error - constructAffineTransform: Must have same number of coordinates");
            }
            return null;
        }
        double[] m = new double[6];
        double[] ATA = new double[9];
        double[] ATc = new double[3];
        int[] indx = new int[3];
        int[] flag = new int[1];
        int len = x.length;
        ATA[0] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[0] = ATA[0] + x[ii] * x[ii];
        }
        ATA[1] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[1] = ATA[1] + x[ii] * y[ii];
        }
        ATA[3] = ATA[1];
        ATA[2] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[2] = ATA[2] + x[ii];
        }
        ATA[6] = ATA[2];
        ATA[4] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[4] = ATA[4] + y[ii] * y[ii];
        }
        ATA[5] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[5] = ATA[5] + y[ii];
        }
        ATA[7] = ATA[5];
        ATA[8] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATA[8] = ATA[8] + 1.0;
        }
        RmaMath.luDComp(ATA, 3, indx, flag);
        ATc[0] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[0] = ATc[0] + x[ii] * xp[ii];
        }
        ATc[1] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[1] = ATc[1] + y[ii] * xp[ii];
        }
        ATc[2] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[2] = ATc[2] + xp[ii];
        }
        RmaMath.luBackSub(ATA, 3, indx, ATc);
        m[0] = ATc[0];
        m[2] = ATc[1];
        m[4] = ATc[2];
        ATc[0] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[0] = ATc[0] + x[ii] * yp[ii];
        }
        ATc[1] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[1] = ATc[1] + y[ii] * yp[ii];
        }
        ATc[2] = 0.0;
        for (ii = 0; ii < len; ++ii) {
            ATc[2] = ATc[2] + yp[ii];
        }
        RmaMath.luBackSub(ATA, 3, indx, ATc);
        m[1] = ATc[0];
        m[3] = ATc[1];
        m[5] = ATc[2];
        return m;
    }

    public static boolean luDComp(double[] a, int n, int[] indx, int[] d) {
        int j;
        double big;
        int i;
        int imax = 0;
        double[] vv = new double[n];
        d[0] = 1;
        for (i = 0; i < n; ++i) {
            big = 0.0;
            for (j = 0; j < n; ++j) {
                double temp = Math.abs(a[i * n + j]);
                if (!(temp > big)) continue;
                big = temp;
            }
            if (big == 0.0) {
                if (LOGGER.isLoggable(Level.INFO)) {
                    LOGGER.log(Level.INFO, "ludcomp: Error-singular matrix");
                }
                return false;
            }
            vv[i] = 1.0 / big;
        }
        for (j = 0; j < n; ++j) {
            double dum;
            int k;
            double sum;
            for (i = 0; i < j; ++i) {
                sum = a[i * n + j];
                for (k = 0; k < i; ++k) {
                    sum -= a[i * n + k] * a[k * n + j];
                }
                a[i * n + j] = sum;
            }
            big = 0.0;
            for (i = j; i < n; ++i) {
                sum = a[i * n + j];
                for (k = 0; k < j; ++k) {
                    sum -= a[i * n + k] * a[k * n + j];
                }
                a[i * n + j] = sum;
                dum = vv[i] * Math.abs(sum);
                if (!(dum >= big)) continue;
                big = dum;
                imax = i;
            }
            if (j != imax) {
                for (k = 0; k < n; ++k) {
                    dum = a[imax * n + k];
                    a[imax * n + k] = a[j * n + k];
                    a[j * n + k] = dum;
                }
                d[0] = d[0] * -1;
                vv[imax] = vv[j];
            }
            indx[j] = imax;
            if (a[j * n + j] == 0.0) {
                if (LOGGER.isLoggable(Level.INFO)) {
                    LOGGER.log(Level.INFO, "ludcomp: Error-singular matrix");
                }
                return false;
            }
            if (j == n - 1) continue;
            dum = 1.0 / a[j * n + j];
            for (i = j + 1; i < n; ++i) {
                int n2 = i * n + j;
                a[n2] = a[n2] * dum;
            }
        }
        return true;
    }

    public static void luBackSub(double[] a, int n, int[] indx, double[] b) {
        int j;
        int i;
        int ii = -1;
        double sum = 0.0;
        for (i = 0; i < n; ++i) {
            int ip = indx[i];
            sum = b[ip];
            b[ip] = b[i];
            if (ii >= 0) {
                for (j = ii; j <= i - 1; ++j) {
                    sum -= a[i * n + j] * b[j];
                }
            } else if (sum != 0.0) {
                ii = i;
            }
            b[i] = sum;
        }
        for (i = n - 1; i >= 0; --i) {
            sum = b[i];
            for (j = i + 1; j < n; ++j) {
                sum -= a[i * n + j] * b[j];
            }
            b[i] = sum / a[i * n + i];
        }
    }

    public static boolean invertMatrix(double[] a) {
        int i;
        if (a == null || a.length < 0) {
            return false;
        }
        int n = (int)Math.sqrt(a.length);
        if (n * n != a.length) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "invertMatrix: Error - input matrix is not square");
            }
            return false;
        }
        int[] indx = new int[n];
        int[] d = new int[1];
        double[] col = new double[n];
        double[] y = new double[n * n];
        boolean brtn = RmaMath.luDComp(a, n, indx, d);
        if (!brtn) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "invertMatrix: Error - luDComp failed");
            }
            return false;
        }
        for (int j = 0; j < n; ++j) {
            for (i = 0; i < n; ++i) {
                col[i] = 0.0;
            }
            col[j] = 1.0;
            RmaMath.luBackSub(a, n, indx, col);
            for (i = 0; i < n; ++i) {
                y[i * n + j] = col[i];
            }
        }
        for (i = 0; i < n * n; ++i) {
            a[i] = y[i];
        }
        return true;
    }

    public static boolean matTranspose(double[] m, int nr, int nc) {
        if (m == null || m.length < 1) {
            return false;
        }
        double[] a = (double[])m.clone();
        for (int j = 0; j < nc; ++j) {
            for (int i = 0; i < nr; ++i) {
                m[j * nc + i] = a[i * nc + j];
            }
        }
        return true;
    }

    public static boolean matMultAS(double[] m, double s, int nr, int nc) {
        if (m == null) {
            return false;
        }
        for (int i = 0; i < nr; ++i) {
            for (int j = 0; j < nc; ++j) {
                m[j + i * nc] = m[j + i * nc] * s;
            }
        }
        return true;
    }

    public static boolean matMultAB(double[] A, double[] B, double[] C, int nra, int nca, int ncb) {
        if (A == null || B == null || C == null) {
            return false;
        }
        if (A.length != nra * nca) {
            return false;
        }
        if (B.length != nca * ncb) {
            return false;
        }
        if (C.length != nra * ncb) {
            return false;
        }
        for (int i = 0; i < nra; ++i) {
            for (int j = 0; j < ncb; ++j) {
                C[i * ncb + j] = 0.0;
                for (int k = 0; k < nca; ++k) {
                    int n = i * ncb + j;
                    C[n] = C[n] + A[i * nca + k] * B[k * ncb + j];
                }
            }
        }
        return true;
    }

    public static boolean matMultAtB(double[] A, double[] B, double[] C, int nra, int nca, int ncb) {
        if (A == null || B == null || C == null) {
            return false;
        }
        if (A.length != nra * nca) {
            return false;
        }
        if (B.length != nra * ncb) {
            return false;
        }
        if (C.length != nca * ncb) {
            return false;
        }
        for (int i = 0; i < nca; ++i) {
            for (int j = 0; j < ncb; ++j) {
                C[i * ncb + j] = 0.0;
                for (int k = 0; k < nra; ++k) {
                    int n = i * ncb + j;
                    C[n] = C[n] + A[k * nca + i] * B[k * ncb + j];
                }
            }
        }
        return true;
    }

    public static boolean matMultABt(double[] A, double[] B, double[] C, int nra, int nca, int nrb) {
        if (A == null || B == null || C == null) {
            return false;
        }
        if (A.length != nra * nca) {
            return false;
        }
        if (B.length != nrb * nca) {
            return false;
        }
        if (C.length != nra * nrb) {
            return false;
        }
        for (int i = 0; i < nra; ++i) {
            for (int j = 0; j < nrb; ++j) {
                C[i * nrb + j] = 0.0;
                for (int k = 0; k < nca; ++k) {
                    int n = i * nrb + j;
                    C[n] = C[n] + A[i * nca + k] * B[j * nca + k];
                }
            }
        }
        return true;
    }

    public static boolean matAddTo(double[] A, double[] B) {
        if (A == null || B == null) {
            return false;
        }
        if (A.length != B.length) {
            return false;
        }
        int imax = A.length;
        for (int i = 0; i < imax; ++i) {
            int n = i;
            B[n] = B[n] + A[i];
        }
        return true;
    }

    public static boolean matSubtTo(double[] A, double[] B) {
        if (A == null || B == null) {
            return false;
        }
        if (A.length != B.length) {
            return false;
        }
        int imax = A.length;
        for (int i = 0; i < imax; ++i) {
            int n = i;
            B[n] = B[n] - A[i];
        }
        return true;
    }

    public static boolean matAddS(double[] A, double S) {
        if (A == null) {
            return false;
        }
        int imax = A.length;
        int i = 0;
        while (i < imax) {
            int n = i++;
            A[n] = A[n] + S;
        }
        return true;
    }

    public static double matTrace(double[] A, int n) {
        if (A == null || A.length != n * n) {
            return Double.NaN;
        }
        double trace = 0.0;
        for (int i = 0; i < n; ++i) {
            trace += A[i * n + i];
        }
        return trace;
    }

    public static boolean matCharPoly(double[] A, double[] coeff, int n) {
        if (A == null || coeff == null) {
            return false;
        }
        if (A.length != n * n || coeff.length != n + 1) {
            return false;
        }
        double[] B = new double[n * n];
        double[] C = new double[n * n];
        Arrays.fill(C, 0.0);
        coeff[0] = 1.0;
        for (int k = 1; k <= n; ++k) {
            B = Arrays.copyOf(C, n * n);
            for (int i = 0; i < n; ++i) {
                int n2 = i * n + i;
                B[n2] = B[n2] + coeff[k - 1];
            }
            RmaMath.matMultAB(A, B, C, n, n, n);
            coeff[k] = -1.0 / (double)k * RmaMath.matTrace(C, n);
        }
        return true;
    }

    public static double matDet(double[] A, int n) {
        if (A == null || A.length != n * n) {
            return Double.NaN;
        }
        double[] coeff = new double[n + 1];
        RmaMath.matCharPoly(A, coeff, n);
        int sgn = n % 2 == 0 ? 1 : -1;
        return (double)sgn * coeff[n];
    }

    public static int matNullity(double[] A, int ra, int ca) {
        return ra - RmaMath.matRank(A, ra, ca);
    }

    public static int matRank(double[] A, int ra, int ca) {
        if (A == null || A.length != ra * ca) {
            return ra;
        }
        if (ra < ca) {
            double[] C = new double[ra * ra];
            RmaMath.matMultABt(A, A, C, ra, ca, ra);
            return RmaMath.matRankPSD(C, ra);
        }
        double[] C = new double[ca * ca];
        RmaMath.matMultAtB(A, A, C, ra, ca, ca);
        return RmaMath.matRankPSD(C, ca);
    }

    public static int matNullityPSD(double[] A, int n) {
        return n - RmaMath.matRankPSD(A, n);
    }

    public static int matRankPSD(double[] A, int n) {
        if (A == null || A.length != n * n) {
            return n;
        }
        double tol = 1.0E-8;
        double scale = RmaMath.matTrace(A, n) / (double)n;
        double scaleInv = Math.abs(scale) < tol ? 0.0 : 1.0 / scale;
        double[] Ascaled = Arrays.copyOf(A, n * n);
        RmaMath.matMultAS(Ascaled, scaleInv, n, n);
        double[] coeff = new double[n + 1];
        RmaMath.matCharPoly(Ascaled, coeff, n);
        for (int i = n; i >= 0; --i) {
            if (!(Math.abs(coeff[i]) > tol)) continue;
            return i;
        }
        return 0;
    }

    public static double roundToSignificantDigits(double value, int significantDigits) {
        double pow = Math.pow(10.0, significantDigits);
        return (double)Math.round(pow * value) / pow;
    }

    public static double log10(double x) {
        return Math.log10(x);
    }

    public static void computeSunriseSunset(int dayOfYear, double latitude, double longitude, TimeZone tz, double[] sunriseContainer, double[] sunsetContainer) {
        double TANB;
        double con1 = 0.01721420632103996;
        double con2 = Math.PI / 180 * latitude;
        double con3 = 57.29577951308232;
        double con4 = 0.40927970959267024;
        double con5 = 0.2617993877991494;
        double con6 = 3.819718634205488;
        int utcOffset = tz.getRawOffset() / 3600000;
        GregorianCalendar calendar = new GregorianCalendar(tz);
        calendar.set(6, dayOfYear);
        if (tz.inDaylightTime(calendar.getTime())) {
            ++utcOffset;
        }
        double deltsl = -1.0 * (longitude - (double)(utcOffset * 15)) / 15.0;
        double rearth = 1.0 + 0.017 * Math.cos(con1 * (186.0 - (double)dayOfYear));
        double declin = con4 * Math.cos(con1 * (172.0 - (double)dayOfYear));
        double RR = rearth * rearth;
        double eqtime = 1.21E-4 - 0.12319 * Math.sin(con1 * ((double)dayOfYear - 1.0) - 0.07014) * -0.16549 * Math.sin(2.0 * con1 * ((double)dayOfYear - 1.0) + 0.3088);
        double declon = Math.abs(declin);
        double TANA = Math.sin(con2) / Math.cos(con2);
        double ACS = TANA * (TANB = Math.sin(declon) / Math.cos(declon));
        if (ACS != 0.0) {
            double XX = Math.sqrt(1.0 - ACS * ACS);
            if ((XX /= ACS) > 0.0) {
                ACS = Math.abs(Math.atan(XX));
                if (declin > 0.0) {
                    ACS = Math.PI - ACS;
                }
            } else {
                ACS = Math.abs(Math.atan(XX));
                if (declin < 0.0) {
                    ACS = Math.PI - ACS;
                }
            }
        } else {
            ACS = 1.5707963267948966;
        }
        double sunrise = 12.0 - con6 * ACS + deltsl;
        double sunset = 24.0 - sunrise + 2.0 * deltsl;
        sunriseContainer[0] = sunrise;
        sunsetContainer[0] = sunset;
    }

    public static int[] decimalToFraction(double decimal) {
        if (decimal == 1.0) {
            return new int[]{1, 1};
        }
        int[] fraction = new int[2];
        double a = decimal - (double)((int)decimal);
        double p = 0.0;
        double q = a;
        while (Math.abs(q - (double)Math.round(q)) > 0.01) {
            q = (p += 1.0) / a;
        }
        fraction[0] = (int)Math.round(q * decimal);
        fraction[1] = (int)Math.round(q);
        return fraction;
    }

    public static boolean equals(double d1, double d2, double tolerance) {
        return (double)Double.compare(d1, d2) == 0.0 || Math.abs(d1 - d2) < tolerance;
    }

    public static double[][] cholesky(double[][] A) {
        int j;
        int i;
        int n = A.length;
        if (n < 1) {
            return new double[0][0];
        }
        double[][] L = new double[n][A[0].length];
        for (i = 0; i < n; ++i) {
            for (j = 0; j < i; ++j) {
                if (A[i][j] != 1.0) continue;
                A[i][j] = 0.9999999999999999;
            }
        }
        for (i = 0; i < n; ++i) {
            L[i] = new double[i + 1];
            for (j = 0; j <= i; ++j) {
                double sum = 0.0;
                for (int k = 0; k < j; ++k) {
                    sum += L[i][k] * L[j][k];
                }
                if (i == j) {
                    L[i][i] = Math.sqrt(A[i][i] - sum);
                    continue;
                }
                L[i][j] = 1.0 / L[j][j] * (A[i][j] - sum);
            }
        }
        for (i = 0; i < n; ++i) {
            for (j = 0; j <= i; ++j) {
                if (L[i][j] > 0.99999) {
                    L[i][j] = 1.0;
                }
                if (!(L[i][j] < 1.0E-5)) continue;
                L[i][j] = 0.0;
            }
        }
        return L;
    }

    public static void main(String[] args) {
        double[] mat = new double[]{1.0, -1.0, 0.0, -1.0, 2.0, -1.0, 0.0, -1.0, 1.0};
        double[] coeff = new double[4];
        RmaMath.matCharPoly(mat, coeff, 3);
        System.out.format("Char poly is %f x^3 + %f x^2 + %f x + %f\n", coeff[0], coeff[1], coeff[2], coeff[3]);
        System.out.format("Rank is %d\n", RmaMath.matRankPSD(mat, 3));
        System.out.format("Rank is %d\n", RmaMath.matRank(mat, 3, 3));
    }
}

